<?php
declare(strict_types=1);

require_once __DIR__ . '/../core.php';

class RedditFetcher implements SiteFetcher {
    public static function canHandle(string $url): bool {
        return preg_match('~(reddit\.com|redd\.it)~i', $url) === 1;
    }

    public static function fetchMedia(string $url): array {
        $apiUrl = self::toJsonUrl($url);
        $token  = redditGetToken(REDDIT_CLIENT_ID, REDDIT_CLIENT_SECRET, REDDIT_USERNAME, REDDIT_PASSWORD);

        $json = httpGet($apiUrl, [
            'Authorization: bearer ' . $token,
            'User-Agent: redownloader/1.0 by reza',
            'Accept: application/json'
        ]);
        $data = json_decode($json, true);
        if (!$data) throw new RuntimeException('Cannot parse Reddit JSON.');

        $post = $data[0]['data']['children'][0]['data']
             ?? $data['data']['children'][0]['data']
             ?? null;
        if (!$post) throw new RuntimeException('Post data not found.');

        $author = sanitizeSlug($post['author'] ?? 'reddit');
        $title  = sanitizeSlug($post['title'] ?? 'reddit-post');
        $list   = [];

        // گالری
        if (!empty($post['is_gallery']) && !empty($post['gallery_data']['items'])) {
            foreach ($post['gallery_data']['items'] as $item) {
                $mid = $item['media_id'] ?? null;
                if (!$mid || empty($post['media_metadata'][$mid])) continue;
                $meta = $post['media_metadata'][$mid];

                $isAnimated = (strtolower($meta['e'] ?? '') === 'animatedimage') || (strtolower($meta['m'] ?? '') === 'image/gif');

                $mp4 = $meta['variants']['mp4']['source']['url']
                    ?? $meta['variants']['mp4']['s']['u']
                    ?? null;

                if ($mp4) {
                    $mp4 = preg_replace('~&amp;~', '&', $mp4);
                    $list[] = [
                        'url'    => $mp4,
                        'type'   => 'video',
                        'author' => $author,
                        'title'  => $title,
                        'ext'    => 'mp4',
                    ];
                    continue;
                }

                $src = $meta['s']['u'] ?? $meta['s']['gif'] ?? null;
                if ($src) {
                    $src = preg_replace('~&amp;~', '&', $src);
                    $ext = self::guessExtFromUrl($src);
                    $list[] = [
                        'url'    => $src,
                        'type'   => $isAnimated ? 'gif' : 'image',
                        'author' => $author,
                        'title'  => $title,
                        'ext'    => $ext,
                    ];
                }
            }
        }

        // لینک تکی
        if (!empty($post['url'])) {
            $u = preg_replace('~&amp;~', '&', $post['url']);
            $ext  = self::guessExtFromUrl($u);
            $type = self::inferTypeFromUrl($u);

            $list[] = [
                'url'    => $u,
                'type'   => $type,
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext,
            ];
        }

        // preview mp4
        if (empty($list) && !empty($post['preview']['images'][0]['variants']['mp4']['source']['url'])) {
            $mp4 = preg_replace('~&amp;~', '&', $post['preview']['images'][0]['variants']['mp4']['source']['url']);
            $list[] = [
                'url'    => $mp4,
                'type'   => 'video',
                'author' => $author,
                'title'  => $title,
                'ext'    => 'mp4',
            ];
        }

        // ویدئو داخلی Reddit
        if (!empty($post['secure_media']['reddit_video']['fallback_url'])) {
            $vurl = preg_replace('~&amp;~', '&', $post['secure_media']['reddit_video']['fallback_url']);
            $ext  = self::guessExtFromUrl($vurl) ?: 'mp4';
            $audioUrl = self::buildRedditAudioUrl($vurl);

            $list[] = [
                'url'       => $vurl,
                'type'      => 'video',
                'author'    => $author,
                'title'     => $title,
                'ext'       => $ext,
                'audio_url' => $audioUrl,
            ];
        }

        if (empty($list)) throw new RuntimeException('No downloadable media found.');
        return $list;
    }

    private static function toJsonUrl(string $url): string {
        $url = preg_replace('~(\?|\#).*$~', '', $url);
        if (!preg_match('~/$~', $url)) $url .= '/';
        $url = preg_replace('~^https://(www\.)?reddit\.com~', 'https://oauth.reddit.com', $url);
        return $url . '.json';
    }

    private static function guessExtFromUrl(string $url): string {
        $path = parse_url($url, PHP_URL_PATH) ?? '';
        $ext  = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        return $ext ?: 'jpg';
    }

    private static function inferTypeFromUrl(string $url): string {
        $ext = strtolower(pathinfo(parse_url($url, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION));
        if (in_array($ext, ['jpg','jpeg','png','webp'])) return 'image';
        if ($ext === 'gif') return 'gif';
        if (in_array($ext, ['mp4','webm','m4v','mov'])) return 'video';
        if (preg_match('~v\.redd\.it|redgifs\.com~i', $url)) return 'video';
        return 'other';
    }

    private static function buildRedditAudioUrl(string $fallbackUrl): ?string {
        $parsed = parse_url($fallbackUrl);
        if (empty($parsed['scheme']) || empty($parsed['host']) || empty($parsed['path'])) return null;
        $path = preg_replace('~/DASH_[^/]+\.mp4$~', '/DASH_audio.mp4', $parsed['path']);
        return $parsed['scheme'] . '://' . $parsed['host'] . $path;
    }
}