<?php
declare(strict_types=1);

require_once __DIR__ . '/../core.php';

class EpornerFetcher implements SiteFetcher {

    public static function canHandle(string $url): bool {
        $host = parse_url($url, PHP_URL_HOST) ?? '';
        $path = parse_url($url, PHP_URL_PATH) ?? '';
        $can = preg_match('~(^|\.)eporner\.com~i', $host)
            && preg_match('~/(gallery|photo)/~i', $path);
        logMessage("canHandle check for $url => " . ($can ? "YES" : "NO"));
        return $can;
    }

    public static function fetchMedia(string $url): array {
        logMessage("Starting fetchMedia for $url");

        $html = httpGet($url, [
            "User-Agent: Mozilla/5.0 (EpornerFetcher/1.0)",
            "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
            "Referer: $url",
        ]);
        logMessage("Fetched HTML from $url, length=" . strlen($html));

        $author = sanitizeSlug(parse_url($url, PHP_URL_HOST) ?? 'eporner');
        $title  = sanitizeSlug(basename(parse_url($url, PHP_URL_PATH) ?? 'gallery'));

        $items = [];

        // اگر لینک photo داده شده باشد
        if (preg_match('~/photo/~i', $url)) {
            logMessage("Detected photo page: $url");

            // عکس اصلی همان صفحه
            $img = self::extractFullImageFromPhotoPage($html, $url, $author, $title);
            if ($img) {
                logMessage("Found FULL image on photo page: " . $img['url']);
                $items[] = $img;
            }

            // لینک گالری مادر را پیدا کن
            if (preg_match('~<a[^>]+href=["\'](/gallery/[^"\']+)["\']~i', $html, $gm)) {
                $galleryUrl = absoluteUrl($url, html_entity_decode($gm[1]));
                logMessage("Found parent gallery link: $galleryUrl");
                $items = array_merge($items, self::fetchGalleryImages($galleryUrl, $author, $title));
            }
        }

        // اگر لینک gallery داده شده باشد
        if (preg_match('~/gallery/~i', $url)) {
            logMessage("Detected gallery page: $url");
            $items = array_merge($items, self::fetchGalleryImages($url, $author, $title));
        }

        // یکتاسازی
        $seen = [];
        $uniq = [];
        foreach ($items as $it) {
            if (!$it || empty($it['url'])) continue;
            if (isset($seen[$it['url']])) continue;
            $seen[$it['url']] = true;
            $uniq[] = $it;
        }

        logMessage("Total unique images found: " . count($uniq));

        if (empty($uniq)) {
            logMessage("❌ No images extracted for $url");
            throw new RuntimeException('❌ هیچ تصویر اصلی در این گالری پیدا نشد.');
        }
        return $uniq;
    }

    // گرفتن همه‌ی عکس‌های گالری
    private static function fetchGalleryImages(string $galleryUrl, string $author, string $title): array {
        logMessage("Fetching gallery images from $galleryUrl");

        $ghtml = httpGet($galleryUrl, [
            "User-Agent: Mozilla/5.0 (EpornerFetcher/1.0)",
            "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
            "Referer: $galleryUrl",
        ]);
        logMessage("Fetched gallery HTML, length=" . strlen($ghtml));

        $photoLinks = self::extractPhotoLinksFromGallery($ghtml, $galleryUrl);
        logMessage("Found " . count($photoLinks) . " photo links in gallery");

        $items = [];
        foreach ($photoLinks as $plink) {
            logMessage("Fetching photo page: $plink");
            $phtml = httpGet($plink, [
                "User-Agent: Mozilla/5.0 (EpornerFetcher/1.0)",
                "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
                "Referer: $galleryUrl",
            ]);
            $img = self::extractFullImageFromPhotoPage($phtml, $plink, $author, $title);
            if ($img) {
                logMessage("Found FULL image in photo page: " . $img['url']);
                $items[] = $img;
            }
        }

        return $items;
    }

    // استخراج عکس اصلی از صفحه‌ی photo
    private static function extractFullImageFromPhotoPage(string $html, string $pageUrl, string $author, string $title): ?array {
        // photohref → لینک اصلی
        if (preg_match('~<a[^>]+class=["\'][^"\']*photohref[^"\']*["\'][^>]+href=["\']([^"\']+)["\']~i', $html, $m)) {
            $fullImg = absoluteUrl($pageUrl, html_entity_decode($m[1]));
            logMessage("Found FULL image via photohref: $fullImg");
            $ext = strtolower(pathinfo(parse_url($fullImg, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION)) ?: 'jpg';
            return [
                'url'    => $fullImg,
                'type'   => 'image',
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext,
            ];
        }

        // JSON-LD contentUrl → لینک اصلی
        if (preg_match('~"contentUrl"\s*:\s*"([^"]+)"~i', $html, $mjson)) {
            $fullImg = absoluteUrl($pageUrl, html_entity_decode($mjson[1]));
            logMessage("Found FULL image via JSON-LD contentUrl: $fullImg");
            $ext = strtolower(pathinfo(parse_url($fullImg, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION)) ?: 'jpg';
            return [
                'url'    => $fullImg,
                'type'   => 'image',
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext,
            ];
        }

        // fallback: mainphoto
        if (preg_match('~<img[^>]+class=["\']mainphoto["\'][^>]+src=["\']([^"\']+)["\']~i', $html, $m2)) {
            $abs = absoluteUrl($pageUrl, html_entity_decode($m2[1]));
            logMessage("Fallback to mainphoto: $abs");
            $ext = strtolower(pathinfo(parse_url($abs, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION)) ?: 'jpg';
            return [
                'url'    => $abs,
                'type'   => 'image',
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext,
            ];
        }

        // fallback: og:image
        if (preg_match('~<meta[^>]+property=["\']og:image["\'][^>]+content=["\']([^"\']+)["\']~i', $html, $m3)) {
            $abs = absoluteUrl($pageUrl, html_entity_decode($m3[1]));
            logMessage("Fallback to og:image: $abs");
            $ext = strtolower(pathinfo(parse_url($abs, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION)) ?: 'jpg';
            return [
                'url'    => $abs,
                'type'   => 'image',
                'author' => $author,
                'title'  => $title,
                'ext'    => $ext,
            ];
        }

        return null;
    }

    // استخراج لینک‌های photo از گالری
    private static function extractPhotoLinksFromGallery(string $html, string $baseUrl): array {
        $links = [];
        // همه‌ی لینک‌های /photo/... داخل گالری
        if (preg_match_all('~<a[^>]+href=["\'](/photo/[^"\']+)["\']~i', $html, $m)) {
            foreach ($m[1] as $href) {
                $links[] = absoluteUrl($baseUrl, html_entity_decode($href));
            }
        }
        return array_values(array_unique($links));
    }
}
