<?php
declare(strict_types=1);

require_once __DIR__ . '/core.php';
require_once __DIR__ . '/fetchers.php'; // رجیستری پلاگین‌ها و getFetcherForUrl

header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$url         = trim($_POST['url'] ?? '');
$contentType = trim($_POST['contentType'] ?? ''); // فقط برای GenericFetcher استفاده می‌شود

if (!$url) {
    http_response_code(400);
    echo json_encode(['error' => '❌ لینک معتبر وارد کنید.']);
    exit;
}

try {
    // انتخاب پلاگین مناسب
    $fetcherClass = getFetcherForUrl($url);

    if (!$fetcherClass) {
        if ($contentType === '') {
            echo json_encode(['requireContentType' => true]);
            exit;
        }
        if (!class_exists('GenericFetcher')) {
            throw new RuntimeException('GenericFetcher در دسترس نیست.');
        }
        $fetcherClass = GenericFetcher::class;
        $medias = $fetcherClass::fetchMedia($url, $contentType);
    } else {
        if ($fetcherClass === GenericFetcher::class) {
            if ($contentType === '') {
                echo json_encode(['requireContentType' => true]);
                exit;
            }
            $medias = $fetcherClass::fetchMedia($url, $contentType);
        } else {
            $medias = $fetcherClass::fetchMedia($url);
        }
    }

    // مسیرهای نویسنده و شناسه لینک
    $author = sanitizeSlug(parse_url($url, PHP_URL_HOST) ?? 'unknown');
    $pathParts = explode('/', trim(parse_url($url, PHP_URL_PATH) ?? '', '/'));
    $linkId = sanitizeSlug(end($pathParts) ?: 'link');
    [$userMediaPath, $userThumbPath, $userMetaPath] = ensureLinkDirs($author, $linkId);

    $progressId = md5($url); // شناسه‌ی یکتا برای گزارش پیشرفت
    $saved = [];
    $downloadedCount = 0;

    foreach ($medias as $m) {
        if (alreadyDownloaded($m['url'])) continue;

        $base   = sanitizeSlug($m['title'] ?? 'media');
        $type   = $m['type'] ?? 'image';
        $ext    = $m['ext'] ?: ($type === 'video' ? 'mp4' : ($type === 'image' ? 'jpg' : 'bin'));
        $hash   = md5($m['url']);

        $name = "{$base}_{$hash}.{$ext}";
        $dest = $userMediaPath . '/' . $name;

        if ($type === 'live') {
            $dummy = $userMediaPath . '/' . "{$base}_{$hash}.txt";
            file_put_contents($dummy, "STREAM: " . $m['url']);
            $thumb = $userThumbPath . '/' . preg_replace('/\.[^.]+$/', '.jpg', $name);
            makePlaceholderThumb($thumb, 'live');
            $saved[] = [
                'file'   => "storage/media/$author/$linkId/" . basename($dummy),
                'thumb'  => "storage/thumbs/$author/$linkId/" . basename($thumb),
                'type'   => 'live',
                'source' => $m['url'],
                'author' => $author,
                'linkId' => $linkId,
                'title'  => $base,
            ];
            continue;
        }

        // دانلود فایل با ثبت پیشرفت
        downloadFile($m['url'], $dest, $progressId);
        $downloadedCount++;

        // ساخت thumbnail
        $thumb = $userThumbPath . '/' . preg_replace('/\.[^.]+$/', '.jpg', $name);
        if ($type === 'image' || $type === 'gif') {
            makeThumb($dest, $thumb);
        } elseif ($type === 'video') {
            $cmd = sprintf(
                'ffmpeg -y -ss 1 -i %s -vframes 1 -vf "scale=320:-1" %s 2>&1',
                escapeshellarg($dest),
                escapeshellarg($thumb)
            );
            exec($cmd, $out, $ret);
            if ($ret !== 0 || !file_exists($thumb)) {
                makePlaceholderThumb($thumb, 'video');
            }
        } else {
            makePlaceholderThumb($thumb, $type);
        }

        $saved[] = [
            'file'   => "storage/media/$author/$linkId/$name",
            'thumb'  => "storage/thumbs/$author/$linkId/" . basename($thumb),
            'type'   => $type,
            'source' => $m['url'],
            'author' => $author,
            'linkId' => $linkId,
            'title'  => $base,
        ];
    }

    // ذخیره متادیتا
    if (!empty($saved)) {
        $postPath = parse_url($url, PHP_URL_PATH) ?? '';
        $postId   = sanitizeSlug($postPath ?: 'post');

        $count = count($saved);
        $size  = 0;
        foreach ($saved as $it) {
            $path = __DIR__ . '/' . $it['file'];
            if (file_exists($path)) $size += filesize($path);
        }

        saveMeta($postId, [
            'source'        => $url,
            'downloaded_at' => date('c'),
            'items'         => $saved,
            'count'         => $count,
            'size'          => $size
        ], $author, $linkId);
    }

    // خروجی JSON برای AJAX
    echo json_encode([
        'success' => true,
        'author'  => $author,
        'linkId'  => $linkId,
        'count'   => count($saved),
        'downloaded' => $downloadedCount,
        'progressId' => $progressId,
        'redirect' => "gallery.php?author={$author}&linkId={$linkId}"
    ]);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
    exit;
}