<?php
declare(strict_types=1);

/* ===========================
   تنظیمات و مسیرها (Config)
   =========================== */

define('STORAGE_PATH', __DIR__ . '/storage');
define('MEDIA_PATH',   STORAGE_PATH . '/media');
define('THUMB_PATH',   STORAGE_PATH . '/thumbs');
define('META_PATH',    STORAGE_PATH . '/meta');

define('CA_BUNDLE', __DIR__ . '/assets/cacert.pem');
define('MAX_DOWNLOADS_PER_POST', 50);
define('TIMEOUT_SECONDS', 400);

// اطلاعات اپلیکیشن Reddit (برای پلاگین Reddit)
define('REDDIT_CLIENT_ID',     'LaX44OR4_eLCYGaKyZiSBg');
define('REDDIT_CLIENT_SECRET', 'ePBp2g-1b0ZPs-4zzFfovYNlVQLkhw');
define('REDDIT_USERNAME',      'fbbongafb');
define('REDDIT_PASSWORD',      'TAn1101101369629');

// ساخت پوشه‌های اصلی
foreach ([STORAGE_PATH, MEDIA_PATH, THUMB_PATH, META_PATH] as $dir) {
    if (!is_dir($dir)) mkdir($dir, 0775, true);
}

/* ===========================
   عیبیابی
   =========================== */
function logMessage(string $message): void {
    $logFile = __DIR__ . '/process.log'; // مسیر فایل لاگ
    $date = date('Y-m-d H:i:s');
    $line = "[$date] $message\n";
    file_put_contents($logFile, $line, FILE_APPEND);
}


/* ===========================
   اینترفیس پلاگین‌ها
   =========================== */

interface SiteFetcher {
    // تشخیص اینکه پلاگین می‌تواند این URL را مدیریت کند
    public static function canHandle(string $url): bool;
    // استخراج مدیا و برگرداندن آرایه استاندارد آیتم‌ها
    // هر آیتم: ['url','type','author','title','ext', optional: 'audio_url']
    public static function fetchMedia(string $url): array;
}

/* ===========================
   توابع کمکی (Helpers)
   =========================== */

function sanitizeSlug(string $input): string {
    $slug = preg_replace('~[^\pL\d]+~u', '-', $input);
    $slug = @iconv('UTF-8', 'ASCII//TRANSLIT', $slug) ?: $input;
    $slug = preg_replace('~[^-\w]+~', '', $slug);
    $slug = trim($slug, '-');
    $slug = preg_replace('~-+~', '-', $slug);
    return strtolower($slug ?: 'post');
}

function absoluteUrl(string $base, string $path): string {
    if ($path === '') return $base;
    if (preg_match('~^https?://~i', $path)) return $path;
    if (strpos($path, '//') === 0) {
        $scheme = parse_url($base, PHP_URL_SCHEME) ?: 'https';
        return $scheme . ':' . $path;
    }
    $parts = parse_url($base);
    $scheme   = $parts['scheme'] ?? 'https';
    $host     = $parts['host'] ?? '';
    $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
    $basePath = $parts['path'] ?? '/';
    if ($path[0] !== '/') {
        if (substr($basePath, -1) !== '/') {
            $basePath = preg_replace('~/[^/]+$~', '/', $basePath);
        }
        $path = $basePath . $path;
    }
    $segments = [];
    foreach (explode('/', $path) as $seg) {
        if ($seg === '' || $seg === '.') continue;
        if ($seg === '..') array_pop($segments);
        else $segments[] = $seg;
    }
    $normalized = '/' . implode('/', $segments);
    return $scheme . '://' . $host . $port . $normalized;
}

function httpGet(string $url, array $headers = []): string {
    $ch = curl_init($url);
    $ref = '';
    $pu = parse_url($url);
    if (!empty($pu['scheme']) && !empty($pu['host'])) {
        $ref = $pu['scheme'] . '://' . $pu['host'];
    }
    $defaultHeaders = ['User-Agent: Mozilla/5.0 (Downloader)', 'Accept: */*'];
    if ($ref) $defaultHeaders[] = 'Referer: ' . $ref;
    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => TIMEOUT_SECONDS,
        CURLOPT_HTTPHEADER     => array_merge($defaultHeaders, $headers),
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Downloader)',
    ];
    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }
    curl_setopt_array($ch, $opts);
    $data = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);
    if ($data === false) throw new RuntimeException("Failed to GET $url: $err");
    return $data;
}

function downloadFile(string $url, string $destPath): void {
    $dir = dirname($destPath);
    if (!is_dir($dir)) mkdir($dir, 0775, true);
    $tmp = $destPath . '.part';
    $fp  = fopen($tmp, 'wb');
    if (!$fp) throw new RuntimeException("Cannot write temp file: $tmp");
    $ch = curl_init($url);
    $ref = '';
    $pu = parse_url($url);
    if (!empty($pu['scheme']) && !empty($pu['host'])) {
        $ref = $pu['scheme'] . '://' . $pu['host'];
    }
    $headers = ['Accept: */*'];
    if ($ref) $headers[] = 'Referer: ' . $ref;
    $opts = [
        CURLOPT_FILE           => $fp,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => TIMEOUT_SECONDS,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Downloader)',
        CURLOPT_HTTPHEADER     => $headers,
    ];
    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }
    curl_setopt_array($ch, $opts);
    $ok  = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    fclose($fp);
    if ($ok === false) {
        @unlink($tmp);
        throw new RuntimeException("Download failed: $err");
    }
    rename($tmp, $destPath);
}

function makeThumb(string $src, string $dest, int $maxW = 320, int $maxH = 320): void {
    $info = @getimagesize($src);
    if (!$info) return;
    [$w, $h, $type] = $info;
    $ratio = min($maxW / $w, $maxH / $h);
    $newW = max(1, (int)($w * $ratio));
    $newH = max(1, (int)($h * $ratio));
    switch ($type) {
        case IMAGETYPE_JPEG: $img = @imagecreatefromjpeg($src); break;
        case IMAGETYPE_PNG:  $img = @imagecreatefrompng($src);  break;
        case IMAGETYPE_GIF:  $img = @imagecreatefromgif($src);  break;
        default: return;
    }
    if (!$img) return;
    $thumb = imagecreatetruecolor($newW, $newH);
    imagecopyresampled($thumb, $img, 0,0,0,0, $newW,$newH, $w,$h);
    @imagejpeg($thumb, $dest, 82);
    imagedestroy($img);
    imagedestroy($thumb);
}

function makePlaceholderThumb(string $dest, string $label = 'FILE'): void {
    $dir = dirname($dest);
    if (!is_dir($dir)) mkdir($dir, 0775, true);
    $img = imagecreatetruecolor(320, 180);
    $bg  = imagecolorallocate($img, 30, 30, 30);
    imagefilledrectangle($img, 0,0,320,180,$bg);
    $txt = imagecolorallocate($img, 200,200,200);
    imagestring($img, 5, 10, 80, strtoupper($label), $txt);
    imagejpeg($img, $dest, 80);
    imagedestroy($img);
}

/* ===========================
   مدیریت متادیتا
   =========================== */

function saveMeta(string $postId, array $meta, string $author = 'unknown', string $linkId = ''): void {
    $author = sanitizeSlug($author);
    $linkId = sanitizeSlug($linkId);
    $dir    = META_PATH . '/' . $author;
    if (!is_dir($dir)) mkdir($dir, 0775, true);

    $file = $dir . '/' . ($linkId ?: $postId) . '.json';
    file_put_contents($file, json_encode($meta, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
}

function loadAllMedia(): array {
    $items   = [];
    $authors = glob(META_PATH . '/*', GLOB_ONLYDIR);

    foreach ($authors as $authorDir) {
        $author = basename($authorDir);
        foreach (glob($authorDir . '/*.json') as $file) {
            $data = json_decode(file_get_contents($file), true);
            if (!$data || empty($data['items'])) continue;

            $linkId = basename($file, '.json');
            $count  = count($data['items']);
            $size   = 0;
            foreach ($data['items'] as $it) {
                if (!empty($it['file'])) {
                    $path = __DIR__ . '/' . $it['file'];
                    if (file_exists($path)) $size += filesize($path);
                }
            }

            foreach ($data['items'] as $it) {
                $items[] = [
                    'path'   => $it['file']   ?? '',
                    'thumb'  => $it['thumb']  ?? '',
                    'type'   => $it['type']   ?? 'image',
                    'name'   => $it['title']  ?? basename($file),
                    'url'    => $it['source'] ?? '',
                    'author' => $author,
                    'linkId' => $linkId,
                    'title'  => $it['title']  ?? '',
                    'ext'    => pathinfo($it['file'] ?? '', PATHINFO_EXTENSION),
                    'count'  => $count,
                    'size'   => $size,
                    'page'   => $data['source'] ?? ''
                ];
            }
        }
    }

    usort($items, fn($a,$b) => strcmp($b['name'], $a['name']));
    return $items;
}

// بررسی دانلود قبلی براساس هش URL
function alreadyDownloaded(string $url): bool {
    $hash    = md5($url);
    $authors = glob(MEDIA_PATH . '/*', GLOB_ONLYDIR);

    foreach ($authors as $authorDir) {
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($authorDir)) as $fileinfo) {
            if ($fileinfo->isFile()) {
                if (strpos($fileinfo->getFilename(), $hash) !== false) {
                    return true;
                }
            }
        }
    }
    return false;
}

/* ===========================
   پوشه‌های نویسنده/لینک
   =========================== */

function ensureLinkDirs(string $author, string $linkId): array {
    $author = sanitizeSlug($author);
    $linkId = sanitizeSlug($linkId);
    $media  = MEDIA_PATH . '/' . $author . '/' . $linkId;
    $thumbs = THUMB_PATH . '/' . $author . '/' . $linkId;
    $meta   = META_PATH  . '/' . $author;

    foreach ([$media, $thumbs, $meta] as $dir) {
        if (!is_dir($dir)) mkdir($dir, 0775, true);
    }
    return [$media, $thumbs, $meta];
}

/* ===========================
   احراز هویت Reddit (برای پلاگین Reddit)
   =========================== */

function redditGetToken(string $client_id, string $client_secret, string $username, string $password): string {
    $ch = curl_init('https://www.reddit.com/api/v1/access_token');
    $opts = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_USERPWD        => $client_id . ':' . $client_secret,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => [
            'grant_type' => 'password',
            'username'   => $username,
            'password'   => $password
        ],
        CURLOPT_HTTPHEADER     => ['User-Agent: MyDownloader/1.0 by reza'],
    ];
    if (file_exists(CA_BUNDLE)) {
        $opts[CURLOPT_SSL_VERIFYPEER] = true;
        $opts[CURLOPT_CAINFO]         = CA_BUNDLE;
    } else {
        $opts[CURLOPT_SSL_VERIFYPEER] = false;
    }

    curl_setopt_array($ch, $opts);
    $response = curl_exec($ch);
    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        throw new RuntimeException('Failed to get token: ' . $err);
    }
    curl_close($ch);

    $data = json_decode($response, true);
    if (empty($data['access_token'])) throw new RuntimeException('No access_token in response: ' . $response);
    return $data['access_token'];
}
